<?php

namespace VM\ApiBundle\Tests\Controller;

use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;

class ApiUserRequestTest extends WebTestCase
{
    protected $email = 'rafal@vm.pl';
    protected $password = '12340987';

    protected $appKey = 'sHug4572RABbO7T6k7zHeQ6oWouCk3';

    public function testUserFullRegistration()
    {
        $client = static::createClient();
        $params = array(
            'name' => 'Rafal',
            'surname' => 'Ignaszewski',
            'email' => $this->email,
            'password' => $this->password
        );

        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_register', $params));

        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('user_id', $responseJson);
        $this->assertArrayHasKey('confirmation_token', $responseJson);

        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_registration_confirm', $responseJson));
        $responseJson = json_decode($client->getResponse()->getContent(), true);
        $this->assertArrayHasKey('success', $responseJson);
    }



    public function testUserLogin()
    {
        $userData = $this->loginUser($this->email, $this->password);

        $this->assertGreaterThan(0, $userData['user_id']);
        $this->assertArrayHasKey('token', $userData);
        $this->assertEquals(strlen($userData['token']), 40);
    }


    public function testUserLoginIncorrect()
    {
        $client = static::createClient();

        $params = array(
            'email' => $this->email,
            'password' => '11234567890'
        );
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_login', $params));

        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('error', $responseJson);
        $this->assertArrayHasKey('code', $responseJson['error']);
        $this->assertArrayHasKey('message', $responseJson['error']);
    }


    public function testUserCheckToken()
    {
        $client = static::createClient();

        $userLoginData = $this->loginUser($this->email, $this->password);

        $this->assertArrayHasKey('user_id', $userLoginData);
        $this->assertArrayHasKey('token', $userLoginData);

        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_check_token', $userLoginData));

        $responseArray = json_decode($client->getResponse()->getContent(), true);
        $this->assertArrayHasKey('token_valid', $responseArray);
        $this->assertTrue($responseArray['token_valid']);
    }


    public function testUserRegistrationEmailExist()
    {
        $client = static::createClient();

        $params = array(
            'name' => 'jan',
            'surname' => 'Kowalski',
            'email' => $this->email,
            'password' => '13424589723645'
        );
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_register', $params));

        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('error', $responseJson);
        $this->assertArrayHasKey('code', $responseJson['error']);
        $this->assertEquals(20003, $responseJson['error']['code']);
    }


    public function testUserRegistrationEmptyEmail()
    {
        $client = static::createClient();


        $params = array(
            'name' => 'jan',
            'surname' => 'Kowalski',
            'email' => '',
            'password' => '13424589723645'
        );
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_register', $params));

        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('error', $responseJson);
        $this->assertArrayHasKey('code', $responseJson['error']);
        $this->assertArrayHasKey('message', $responseJson['error']);
        $this->assertEquals(20003, $responseJson['error']['code']);
    }

    public function testUserRegistrationEmptyNameAndSurname()
    {
        $client = static::createClient();


        $params = array(
            'name' => '',
            'surname' => '',
            'email' => 'jan@vm.pl',
            'password' => '13424589723645'
        );
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_register', $params));

        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('error', $responseJson);
        $this->assertArrayHasKey('code', $responseJson['error']);
        $this->assertArrayHasKey('message', $responseJson['error']);
        $this->assertEquals(20003, $responseJson['error']['code']);
    }

    public function testUserRegistrationEmptyPassword()
    {
        $client = static::createClient();
        $params = array(
            'name' => 'jan',
            'surname' => 'Kowalski',
            'email' => 'jan@vm.pl',
            'password' => ''
        );
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_register', $params));

        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('error', $responseJson);
        $this->assertArrayHasKey('code', $responseJson['error']);
        $this->assertArrayHasKey('message', $responseJson['error']);
        $this->assertEquals(20003, $responseJson['error']['code']);
    }



    public function testUserUpdate()
    {
        $userData = $this->loginUser($this->email, $this->password);

        $client = static::createClient();

        $userData['data'] = array(
            'name' => 'Tomasz',
            'surname' => 'Kowalski'
        );
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_update', $userData));

        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('success', $responseJson);
        $this->assertTrue($responseJson['success']);
    }


    public function testUserUpdateChangePassword()
    {
        $userData = $this->loginUser($this->email, $this->password);

        $client = static::createClient();

        $userData['data'] = array(
            'password' => '1123456789'
        );
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_update', $userData));

        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('success', $responseJson);
        $this->assertTrue($responseJson['success']);

        // rollback password
        $userData['data'] = array(
            'password' => $this->password
        );
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_update', $userData));

        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('success', $responseJson);
        $this->assertTrue($responseJson['success']);
    }


    public function testUserUpdateNotFullRequestedData()
    {
        $userData = $this->loginUser($this->email, $this->password);
        $client = static::createClient();

        $userData['data'] = array(
        );
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_update', $userData));
        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('error', $responseJson);
        $this->assertArrayHasKey('code', $responseJson['error']);
        $this->assertArrayHasKey('message', $responseJson['error']);
    }


    public function testUserUpdateWithEmptyName()
    {
        $userData = $this->loginUser($this->email, $this->password);
        $client = static::createClient();

        $userData['data'] = array(
            'name' => '',
            'surname' => 'Kowalski',
            'password' => '123'
        );
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_update', $userData));
        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('error', $responseJson);
        $this->assertArrayHasKey('code', $responseJson['error']);
        $this->assertArrayHasKey('message', $responseJson['error']);
    }


    public function testPasswordRecoveryTokenAndChangePassword()
    {
        $client = static::createClient();

        // password recovery token generation
        $params = array(
            'email' => $this->email
        );
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_generate_password_recovery_token', $params));

        $responseJson = json_decode($client->getResponse()->getContent(), true);
        $this->assertArrayHasKey('password_recovery_token', $responseJson);
        $passwordRecoveryToken = $responseJson['password_recovery_token'];

        // change password
        $params = array(
            'email' => $this->email,
            'password_recovery_token' => $passwordRecoveryToken,
            'password' => '12345'
        );
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_change_password', $params));

        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('success', $responseJson);
        $this->assertTrue($responseJson['success']);

        // login with new password
        $userData = $this->loginUser($this->email, '12345');
        $this->assertArrayHaskey('user_id', $userData);


        // change password to old

        $userData['data'] = array(
            'password' => $this->password
        );
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_update', $userData));

        $responseJson = json_decode($client->getResponse()->getContent(), true);
        $this->assertArrayHasKey('success', $responseJson);
        $this->assertTrue($responseJson['success']);
    }


    public function testGetUser()
    {
        $userData = $this->loginUser($this->email, $this->password);
        $client = static::createClient();

        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_get', $userData));
        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('user_id', $responseJson);
        $this->assertEquals($responseJson['user_id'], $userData['user_id']);
        $this->assertArrayHasKey('email', $responseJson);
        $this->assertEquals($responseJson['email'], $this->email);
        $this->assertArrayHasKey('name', $responseJson);
        $this->assertEquals($responseJson['name'], 'Tomasz');
        $this->assertArrayHasKey('surname', $responseJson);
        $this->assertEquals($responseJson['surname'], 'Kowalski');
        $this->assertArrayHasKey('last_login', $responseJson);
    }


    public function testUsersGetNames()
    {
        $ids = array();
        $users = array(
            array('user_id' => 0, 'name' => 'Jan', 'surname' => 'Zimny', 'email' => 'jan.zimny@vm.pl'),
            array('user_id' => 0, 'name' => 'Dorota', 'surname' => 'Piękna', 'email' => 'dorota.piekna@vm.pl'),
            array('user_id' => 0, 'name' => 'Czesław', 'surname' => 'Niewiadomski', 'email' => 'czeslaw.niewiadomski@vm.pl'),
        );

        foreach($users as &$userData)
        {
            $client = static::createClient();
            $params = array(
                'name' => $userData['name'],
                'surname' => $userData['surname'],
                'email' => $userData['email'],
                'password' => $this->password
            );

            $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_register', $params));
            $responseJson = json_decode($client->getResponse()->getContent(), true);
            $this->assertArrayHasKey('user_id', $responseJson);
            $this->assertArrayHasKey('confirmation_token', $responseJson);


            $userData['user_id'] = $responseJson['user_id'];
            $ids[] = $userData['user_id'];
        }

        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_get_names', array('users' => $ids)));
        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('users', $responseJson);
        $this->assertEquals(count($ids), count($responseJson['users']));
    }

    public function testSetAndGetUserRights()
    {
        $userData = $this->loginUser($this->email, $this->password);
        $userData['application'] = 'GAME';


        $client = static::createClient();
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_get_roles', $userData));
        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('roles', $responseJson);
        $this->assertEquals(0, count($responseJson['roles']));

        $setRolesData = $userData;
        $setRolesData['roles'] = array('ROLE_USER', 'ROLE_TEACHER');
//        var_dump($this->getJsonString('user_set_roles', $setRolesData));
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_set_roles', $setRolesData));
        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('success', $responseJson);
        $this->assertTrue($responseJson['success']);

        $client = static::createClient();
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_get_roles', $userData));
        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('roles', $responseJson);
        $this->assertEquals(2, count($responseJson['roles']));

        return $responseJson;
    }

    public function testApplicationFunctions()
    {
        $userData = $this->loginUser($this->email, $this->password);

        $client = static::createClient();
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_app_update', array('user_id' => $userData['user_id'], 'data' => array('name' => 'Noweimie'))));
        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('success', $responseJson);
        $this->assertTrue($responseJson['success']);


        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_app_delete', array('user_id' => $userData['user_id'])));
        $responseJson = json_decode($client->getResponse()->getContent(), true);

        $this->assertArrayHasKey('success', $responseJson);
        $this->assertTrue($responseJson['success']);

    }

    public function testLoginUserAndTeacher()
    {
        $userData = $this->loginUser('uczen5@vm.pl', md5('uczen5'));
        $userData = $this->loginUser('nauczyciel@vm.pl', md5('nauczyciel'));
    }



    public function testLoginWithDevice()
    {
        $userData = $this->loginUser('nauczyciel@vm.pl', md5('nauczyciel'), 'SOME_DEVICE_ID');
        sleep(3);
        $userData = $this->loginUser('nauczyciel@vm.pl', md5('nauczyciel'), 'SOME_DEVICE_ID');
        sleep(1);
        $userData = $this->loginUser('nauczyciel@vm.pl', md5('nauczyciel'), 'SOME_DEVICE_ID_2');
        sleep(1);
        $userData = $this->loginUser('nauczyciel@vm.pl', md5('nauczyciel'), 'SOME_DEVICE_ID_2');
        sleep(1);
        $userData = $this->loginUser('nauczyciel@vm.pl', md5('nauczyciel'), 'SOME_DEVICE_ID_3');
        sleep(1);
        $userLoginData = $this->loginUser('nauczyciel@vm.pl', md5('nauczyciel'), 'SOME_DEVICE_ID_3');
        sleep(1);
        $userData = $this->loginUser('nauczyciel@vm.pl', md5('nauczyciel'), 'SOME_DEVICE_ID_2');
        sleep(1);
        $userData = $this->loginUser('nauczyciel@vm.pl', md5('nauczyciel'));
        sleep(1);
        $userData = $this->loginUser('nauczyciel@vm.pl', md5('nauczyciel'), 'SOME_DEVICE_ID_4');
        sleep(1);
        $userData = $this->loginUser('nauczyciel@vm.pl', md5('nauczyciel'));
        sleep(2);
        $userData = $this->loginUser('nauczyciel@vm.pl', md5('nauczyciel'));

        $client = static::createClient();
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_check_token', $userLoginData));

        $responseArray = json_decode($client->getResponse()->getContent(), true);
        $this->assertArrayHasKey('token_valid', $responseArray);
        $this->assertTrue($responseArray['token_valid']);
    }


    public function testLoginAndLogout()
    {
        $userLoginData = $this->loginUser('nauczyciel@vm.pl', md5('nauczyciel'), 'SOME_DEVICE_ID_6');

        sleep(2);
        $client = static::createClient();
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $this->getJsonString('user_logout', $userLoginData));

        $responseArray = json_decode($client->getResponse()->getContent(), true);
        $this->assertArrayHasKey('logout', $responseArray);
        $this->assertTrue($responseArray['logout']);
    }






    protected function loginUser($email, $password, $device = '')
    {
        $client = static::createClient();

        if(empty($device))
        {
            $contentJson = '{"appKey": "sHug4572RABbO7T6k7zHeQ6oWouCk3", "method": "user_login", "params": {"email": "' . $email. '", "password": "'. $password . '"}}';
        }
        else
        {
            $contentJson = '{"appKey": "sHug4572RABbO7T6k7zHeQ6oWouCk3", "method": "user_login", "params": {"email": "' . $email. '", "password": "'. $password . '", "deviceID": "' . $device . '"}}';
        }
        $client->request('POST', '/api', array(), array(), array('CONTENT_TYPE' => 'application/json'), $contentJson);

        $responseJson = json_decode($client->getResponse()->getContent(), true);
        $this->assertArrayHasKey('user_id', $responseJson);
        return $responseJson;
    }


    protected function getJsonString($method, $params)
    {
        $array = array(
            'appKey' => $this->appKey,
            'method' => $method,
            'params' => $params
        );

        return json_encode($array);
    }

}